<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using staff_payments/view.php version: e3f4a5b6-c7d8-4e9f-a0b1-2c3d4e5f6a7b");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/staff_payments_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Get staff_id
$staff_id = $_GET['staff_id'] ?? '';
if (empty($staff_id)) {
    $_SESSION['error'] = 'Invalid staff ID.';
    header("Location: list.php");
    exit;
}

// Fetch staff details
try {
    $stmt = $conn->prepare("
        SELECT s.staff_id, s.first_name, s.last_name, s.monthly_salary
        FROM staff s
        WHERE s.staff_id = ? AND s.status = 'active'
    ");
    $stmt->bind_param('s', $staff_id);
    $stmt->execute();
    $staff = $stmt->get_result()->fetch_assoc();
    if (!$staff) {
        $_SESSION['error'] = 'Staff not found or inactive.';
        header("Location: list.php");
        exit;
    }
} catch (Exception $e) {
    error_log("[$timestamp] Fetch staff failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch staff failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load staff details.';
    header("Location: list.php");
    exit;
}

// Get date filters
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');

// Validate dates
if ($start_date > $end_date) {
    $_SESSION['error'] = 'Start date must be before or equal to end date.';
    $start_date = date('Y-m-01');
    $end_date = date('Y-m-t');
}

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="staff_payments_loans_' . $staff_id . '_' . date('Ymd') . '.csv"');

    $output = fopen('php://output', 'w');
    // Payments header
    fputcsv($output, ['Payment History for ' . $staff['first_name'] . ' ' . $staff['last_name']]);
    fputcsv($output, ['Payment ID', 'Amount (KES)', 'Type', 'Date', 'Bank Account', 'Reference', 'Created At']);
    foreach ($payments as $payment) {
        fputcsv($output, [
            $payment['payment_id'],
            number_format($payment['amount'], 2),
            $payment['payment_type'],
            $payment['payment_date'],
            $payment['bank_account_no'],
            $payment['reference_no'],
            $payment['created_at']
        ]);
    }
    // Loans header
    fputcsv($output, []);
    fputcsv($output, ['Loan History for ' . $staff['first_name'] . ' ' . $staff['last_name']]);
    fputcsv($output, ['Loan ID', 'Amount (KES)', 'Monthly Deduction (KES)', 'Status', 'Requested At', 'Approved At']);
    foreach ($loans as $loan) {
        fputcsv($output, [
            $loan['loan_id'],
            number_format($loan['amount'], 2),
            number_format($loan['monthly_deduction'], 2),
            $loan['status'],
            $loan['requested_at'],
            $loan['approved_at'] ?? 'N/A'
        ]);
    }
    fclose($output);
    exit;
}

// Fetch payments
$payments = [];
try {
    $stmt = $conn->prepare("
        SELECT payment_id, amount, payment_type, payment_date, bank_account_no, reference_no, created_at
        FROM staff_payments
        WHERE staff_id = ? AND payment_date BETWEEN ? AND ?
        ORDER BY payment_date DESC
    ");
    $stmt->bind_param('sss', $staff_id, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $payments[] = $row;
    }
    error_log("[$timestamp] Fetched " . count($payments) . " payments for staff ID $staff_id");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($payments) . " payments for staff ID $staff_id\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch payments failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch payments failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load payments.';
}

// Fetch loans
$loans = [];
try {
    $stmt = $conn->prepare("
        SELECT loan_id, amount, monthly_deduction, status, requested_at, approved_at
        FROM loans
        WHERE staff_id = ?
        ORDER BY requested_at DESC
    ");
    $stmt->bind_param('s', $staff_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $loans[] = $row;
    }
    error_log("[$timestamp] Fetched " . count($loans) . " loans for staff ID $staff_id");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($loans) . " loans for staff ID $staff_id\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch loans failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch loans failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load loans.';
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment & Loan History - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); border: 1px solid #e0e0e0; }
        .filter-card { background: linear-gradient(to bottom, #007bff, #0056b3); color: white; }
        .filter-card .card-body { background: white; border-radius: 0 0 10px 10px; }
        .btn-primary, .btn-success { background-color: #007bff; border-color: #007bff; transition: all 0.3s ease; }
        .btn-primary:hover, .btn-success:hover { background-color: #0056b3; border-color: #0056b3; box-shadow: 0 2px 8px rgba(0,0,0,0.2); }
        .form-label { font-weight: 500; color: #333; margin-bottom: 8px; }
        .form-control { border: 1px solid #007bff; border-radius: 5px; transition: border-color 0.3s ease; }
        .form-control:hover { border-color: #0056b3; }
        .alert { border-radius: 8px; }
        .table { font-size: 0.9rem; }
        .table-responsive { max-height: 400px; overflow-y: auto; }
        .filter-btn, .export-btn { border: none; border-radius: 5px; padding: 10px 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.2); transition: all 0.3s ease; }
        .filter-btn { background: linear-gradient(to right, #007bff, #00aaff); }
        .filter-btn:hover { background: linear-gradient(to right, #0056b3, #0088cc); box-shadow: 0 4px 12px rgba(0,0,0,0.3); }
        .export-btn { background: linear-gradient(to right, #28a745, #20c997); }
        .export-btn:hover { background: linear-gradient(to right, #218838, #1c8b73); box-shadow: 0 4px 12px rgba(0,0,0,0.3); }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Payment & Loan History for <?php echo htmlspecialchars($staff['first_name'] . ' ' . $staff['last_name']); ?></h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <div class="card filter-card mb-4">
            <div class="card-body p-4">
                <form method="GET" class="row g-3">
                    <input type="hidden" name="staff_id" value="<?php echo htmlspecialchars($staff_id); ?>">
                    <div class="col-md-5">
                        <label for="start_date" class="form-label">Start Date</label>
                        <input type="date" name="start_date" id="start_date" class="form-control" value="<?php echo htmlspecialchars($start_date); ?>" required>
                    </div>
                    <div class="col-md-5">
                        <label for="end_date" class="form-label">End Date</label>
                        <input type="date" name="end_date" id="end_date" class="form-control" value="<?php echo htmlspecialchars($end_date); ?>" required>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn filter-btn w-100"><i class="fas fa-filter"></i> Filter</button>
                    </div>
                </form>
            </div>
        </div>
        <div class="mb-4">
            <a href="?staff_id=<?php echo urlencode($staff_id); ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&export=csv" class="btn export-btn"><i class="fas fa-download"></i> Export to CSV</a>
        </div>
        <div class="card mb-4">
            <div class="card-body">
                <h5>Payment History</h5>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>Payment ID</th>
                                <th>Amount (KES)</th>
                                <th>Type</th>
                                <th>Date</th>
                                <th>Bank Account</th>
                                <th>Reference</th>
                                <th>Created At</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($payments)): ?>
                                <tr><td colspan="7" class="text-center">No payments found for the selected period.</td></tr>
                            <?php else: ?>
                                <?php foreach ($payments as $payment): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($payment['payment_id']); ?></td>
                                        <td>KSh <?php echo number_format($payment['amount'], 2); ?></td>
                                        <td><?php echo htmlspecialchars($payment['payment_type']); ?></td>
                                        <td><?php echo htmlspecialchars($payment['payment_date']); ?></td>
                                        <td><?php echo htmlspecialchars($payment['bank_account_no']); ?></td>
                                        <td><?php echo htmlspecialchars($payment['reference_no']); ?></td>
                                        <td><?php echo htmlspecialchars($payment['created_at']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <div class="card">
            <div class="card-body">
                <h5>Loan History</h5>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>Loan ID</th>
                                <th>Amount (KES)</th>
                                <th>Monthly Deduction (KES)</th>
                                <th>Status</th>
                                <th>Requested At</th>
                                <th>Approved At</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($loans)): ?>
                                <tr><td colspan="6" class="text-center">No loans found.</td></tr>
                            <?php else: ?>
                                <?php foreach ($loans as $loan): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($loan['loan_id']); ?></td>
                                        <td>KSh <?php echo number_format($loan['amount'], 2); ?></td>
                                        <td>KSh <?php echo number_format($loan['monthly_deduction'], 2); ?></td>
                                        <td><?php echo htmlspecialchars($loan['status']); ?></td>
                                        <td><?php echo htmlspecialchars($loan['requested_at']); ?></td>
                                        <td><?php echo htmlspecialchars($loan['approved_at'] ?? 'N/A'); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>