<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';
?>
<div class="container">
    <!-- Dashboard Section -->
    <section id="dashboard" class="mt-4">
        <h2>Inventory & Promotions Dashboard</h2>
        <div class="card">
            <div class="card-body">
                <table class="table table-bordered">
                    <thead>
                        <tr>
                            <th>Item Code</th>
                            <th>Description</th>
                            <th>Current Price</th>
                            <th>Stock</th>
                            <th>Offer Price</th>
                            <th>Promotion Status</th>
                            <th>Price Change Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody id="inventory-table-body">
                        <!-- Data will be populated by JavaScript -->
                    </tbody>
                </table>
            </div>
        </div>
    </section>

    <!-- Add/Edit Record Section -->
    <section id="add-record" class="mt-4">
        <h2>Add/Edit Inventory Record</h2>
        <div class="card">
            <div class="card-body">
                <form id="inventory-form">
                    <input type="hidden" id="record-id">
                    <div class="row">
                        <!-- Inventory Fields -->
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Item Code</label>
                                <input type="text" id="item_code" class="form-control" placeholder="ITEM001" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Description</label>
                                <textarea id="description" class="form-control" placeholder="Product description" required></textarea>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Original Price</label>
                                <input type="number" step="0.01" id="original_price" class="form-control" placeholder="100.00" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Current Price</label>
                                <input type="number" step="0.01" id="current_price" class="form-control" placeholder="99.99" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Stock Quantity</label>
                                <input type="number" id="stock_quantity" class="form-control" placeholder="50" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Supplier ID</label>
                                <input type="number" id="supplier_id" class="form-control" placeholder="1">
                            </div>
                        </div>
                        <!-- Promotion Fields -->
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Offer Price</label>
                                <input type="number" step="0.01" id="offer_price" class="form-control" placeholder="89.99" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Promotion Start Date</label>
                                <input type="date" id="promotion_start_date" class="form-control" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Promotion End Date</label>
                                <input type="date" id="promotion_end_date" class="form-control" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Promotion Status</label>
                                <select id="promotion_status" class="form-control" required>
                                    <option value="Pending">Pending</option>
                                    <option value="Active">Active</option>
                                    <option value="Expired">Expired</option>
                                </select>
                            </div>
                        </div>
                        <!-- Price Change Fields -->
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Old Price</label>
                                <input type="number" step="0.01" id="old_price" class="form-control" placeholder="100.00" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>New Price</label>
                                <input type="number" step="0.01" id="new_price" class="form-control" placeholder="99.99" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Price Change Status</label>
                                <select id="price_change_status" class="form-control" required>
                                    <option value="Pending">Pending</option>
                                    <option value="Approved">Approved</option>
                                    <option value="Rejected">Rejected</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Price Change Reason</label>
                                <textarea id="price_change_reason" class="form-control" placeholder="Reason for price change"></textarea>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Price Change Approved By</label>
                                <input type="number" id="price_change_approved_by" class="form-control" placeholder="User ID">
                            </div>
                        </div>
                        <!-- Inventory Log Fields -->
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Change Description</label>
                                <textarea id="change_description" class="form-control" placeholder="Description of change"></textarea>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Changed By</label>
                                <input type="number" id="changed_by" class="form-control" placeholder="User ID">
                            </div>
                        </div>
                        <!-- Submit Button -->
                        <div class="col-md-12">
                            <button type="submit" class="btn btn-primary">Save Record</button>
                            <button type="button" id="cancel-edit" class="btn btn-secondary hidden">Cancel</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </section>

    <!-- Bulk Upload Section -->
    <section id="bulk-upload" class="mt-4">
        <h2>Bulk Upload Inventory Records</h2>
        <div class="card">
            <div class="card-body">
                <p class="text-muted">Upload a CSV file with headers matching the inventory fields (e.g., item_code, description, original_price, etc.).</p>
                <input type="file" id="csv-file" accept=".csv" class="form-control-file mb-2">
                <button onclick="uploadCSV()" class="btn btn-primary">Upload CSV</button>
                <p id="upload-error" class="alert alert-danger mt-2 hidden">Error: Invalid CSV format or data.</p>
            </div>
        </div>
    </section>
</div>

<script>
    // Load data into the table
    async function loadTableData() {
        try {
            const response = await fetch('api.php?action=get', {
                headers: { 'Accept': 'application/json' }
            });
            const text = await response.text();
            console.log('Raw API response:', text);
            
            let data;
            try {
                data = JSON.parse(text);
            } catch (e) {
                throw new Error('Invalid JSON response: ' + e.message);
            }
            
            if (!data.success) throw new Error(data.error);
            
            const tableBody = document.getElementById("inventory-table-body");
            tableBody.innerHTML = "";
            data.records.forEach(item => {
                const row = document.createElement("tr");
                row.innerHTML = `
                    <td>${item.item_code}</td>
                    <td>${item.description}</td>
                    <td>$${parseFloat(item.current_price).toFixed(2)}</td>
                    <td>${item.stock_quantity}</td>
                    <td>$${parseFloat(item.offer_price).toFixed(2)}</td>
                    <td>${item.promotion_status}</td>
                    <td>${item.price_change_status}</td>
                    <td>
                        <button onclick="editRecord(${item.id})" class="btn btn-sm btn-info">Edit</button>
                        <button onclick="deleteRecord(${item.id})" class="btn btn-sm btn-danger">Delete</button>
                    </td>
                `;
                tableBody.appendChild(row);
            });
        } catch (err) {
            console.error("Error loading table data:", err);
            alert("Failed to load inventory data: " + err.message);
        }
    }

    // Form submission handler
    document.getElementById("inventory-form").addEventListener("submit", async function (e) {
        e.preventDefault();
        const record = {
            id: parseInt(document.getElementById("record-id").value) || 0,
            item_code: document.getElementById("item_code").value,
            description: document.getElementById("description").value,
            original_price: parseFloat(document.getElementById("original_price").value),
            current_price: parseFloat(document.getElementById("current_price").value),
            stock_quantity: parseInt(document.getElementById("stock_quantity").value),
            supplier_id: parseInt(document.getElementById("supplier_id").value) || null,
            offer_price: parseFloat(document.getElementById("offer_price").value),
            promotion_start_date: document.getElementById("promotion_start_date").value,
            promotion_end_date: document.getElementById("promotion_end_date").value,
            promotion_status: document.getElementById("promotion_status").value,
            old_price: parseFloat(document.getElementById("old_price").value),
            new_price: parseFloat(document.getElementById("new_price").value),
            price_change_status: document.getElementById("price_change_status").value,
            price_change_reason: document.getElementById("price_change_reason").value || null,
            price_change_approved_by: parseInt(document.getElementById("price_change_approved_by").value) || null,
            change_description: document.getElementById("change_description").value || null,
            changed_by: parseInt(document.getElementById("changed_by").value) || null
        };

        try {
            const response = await fetch('api.php?action=save', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'Accept': 'application/json' },
                body: JSON.stringify(record)
            });
            const text = await response.text();
            console.log('Save API response:', text);
            
            const result = JSON.parse(text);
            if (!result.success) throw new Error(result.error);

            await loadTableData();
            resetForm();
            window.location.hash = "dashboard";
        } catch (err) {
            console.error("Error saving record:", err);
            alert("Failed to save record: " + err.message);
        }
    });

    // Edit record
    async function editRecord(id) {
        try {
            const response = await fetch(`api.php?action=get_by_id&id=${id}`, {
                headers: { 'Accept': 'application/json' }
            });
            const text = await response.text();
            console.log('Edit API response:', text);
            
            const result = JSON.parse(text);
            if (!result.success) throw new Error(result.error);

            const record = result.record;
            document.getElementById("record-id").value = record.id;
            document.getElementById("item_code").value = record.item_code;
            document.getElementById("description").value = record.description;
            document.getElementById("original_price").value = record.original_price;
            document.getElementById("current_price").value = record.current_price;
            document.getElementById("stock_quantity").value = record.stock_quantity;
            document.getElementById("supplier_id").value = record.supplier_id || "";
            document.getElementById("offer_price").value = record.offer_price;
            document.getElementById("promotion_start_date").value = record.promotion_start_date;
            document.getElementById("promotion_end_date").value = record.promotion_end_date;
            document.getElementById("promotion_status").value = record.promotion_status;
            document.getElementById("old_price").value = record.old_price;
            document.getElementById("new_price").value = record.new_price;
            document.getElementById("price_change_status").value = record.price_change_status;
            document.getElementById("price_change_reason").value = record.price_change_reason || "";
            document.getElementById("price_change_approved_by").value = record.price_change_approved_by || "";
            document.getElementById("change_description").value = record.change_description || "";
            document.getElementById("changed_by").value = record.changed_by || "";
            document.getElementById("cancel-edit").classList.remove("hidden");
            window.location.hash = "add-record";
        } catch (err) {
            console.error("Error loading record for edit:", err);
            alert("Failed to load record: " + err.message);
        }
    }

    // Delete record
    async function deleteRecord(id) {
        if (confirm("Are you sure you want to delete this record?")) {
            try {
                const response = await fetch(`api.php?action=delete&id=${id}`, {
                    method: 'POST',
                    headers: { 'Accept': 'application/json' }
                });
                const text = await response.text();
                console.log('Delete API response:', text);
                
                const result = JSON.parse(text);
                if (!result.success) throw new Error(result.error);

                await loadTableData();
            } catch (err) {
                console.error("Error deleting record:", err);
                alert("Failed to delete record: " + err.message);
            }
        }
    }

    // Reset form
    function resetForm() {
        document.getElementById("inventory-form").reset();
        document.getElementById("record-id").value = "";
        document.getElementById("cancel-edit").classList.add("hidden");
    }

    // Cancel edit
    document.getElementById("cancel-edit").addEventListener("click", function () {
        resetForm();
        window.location.hash = "dashboard";
    });

    // CSV upload handler
    async function uploadCSV() {
        const fileInput = document.getElementById("csv-file");
        const errorMessage = document.getElementById("upload-error");
        errorMessage.classList.add("hidden");

        if (!fileInput.files[0]) {
            errorMessage.textContent = "Error: Please select a CSV file.";
            errorMessage.classList.remove("hidden");
            return;
        }

        const formData = new FormData();
        formData.append("csv_file", fileInput.files[0]);

        try {
            const response = await fetch('api.php?action=upload_csv', {
                method: 'POST',
                body: formData
            });
            const text = await response.text();
            console.log('CSV Upload API response:', text);
            
            const result = JSON.parse(text);
            if (!result.success) throw new Error(result.error);

            await loadTableData();
            fileInput.value = "";
            window.location.hash = "dashboard";
        } catch (err) {
            errorMessage.textContent = `Error: ${err.message}`;
            errorMessage.classList.remove("hidden");
        }
    }

    // Initial table load
    loadTableData();
</script>