<?php
session_start();
require_once '../../config/db.php';

// Prevent back form resubmission
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) die("Invalid ID.");

$query = "
    SELECT pct.*, b.branch_name, s.name AS supplier_name
    FROM petty_cash_transactions pct
    LEFT JOIN branches b ON pct.branch_id = b.id
    LEFT JOIN suppliers s ON pct.supplier_id = s.id
    WHERE pct.id = ?
";
$stmt = $conn->prepare($query);
$stmt->bind_param('i', $id);
$stmt->execute();
$result = $stmt->get_result();
$transaction = $result->fetch_assoc();

if (!$transaction) die("Transaction not found.");

// Amount in Words
function numberToWords($num) {
    $ones = ["", "One", "Two", "Three", "Four", "Five", "Six",
        "Seven", "Eight", "Nine", "Ten", "Eleven", "Twelve",
        "Thirteen", "Fourteen", "Fifteen", "Sixteen", "Seventeen",
        "Eighteen", "Nineteen"];
    $tens = ["", "", "Twenty", "Thirty", "Forty", "Fifty", "Sixty", "Seventy", "Eighty", "Ninety"];

    if ($num == 0) return "Zero";
    if ($num < 20) return $ones[$num];
    if ($num < 100) return $tens[intval($num / 10)] . (($num % 10) ? " " . $ones[$num % 10] : "");
    if ($num < 1000) return $ones[intval($num / 100)] . " Hundred" . (($num % 100) ? " " . numberToWords($num % 100) : "");
    if ($num < 1000000) return numberToWords(intval($num / 1000)) . " Thousand" . (($num % 1000) ? " " . numberToWords($num % 1000) : "");

    return "Number too large";
}

$whole_amount = intval($transaction['amount']);
$amount_in_words = numberToWords($whole_amount) . " Only";
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Petty Cash Voucher</title>
  <style>
    @page {
      size: 80mm auto;
      margin: 0;
    }
    body {
      width: 80mm;
      margin: 0;
      padding: 0;
      font-family: 'Arial', sans-serif;
      font-size: 15px;
      color: #000;
      text-align: center;
    }
    .container {
      padding: 5px 3px;
      width: 100%;
    }
    h2 {
      margin: 5px 0 10px;
      font-size: 22px;
      font-weight: bold;
    }
    .logo img {
      width: 75px;
      margin-bottom: 5px;
    }
    .section {
      margin: 5px 0;
      font-size: 15px;
    }
    .section div {
      margin-bottom: 5px;
    }
    .description {
      margin-top: 8px;
      font-size: 15px;
      padding: 0 5px;
    }
    .line {
      border-top: 2px dashed #000;
      margin: 10px 0;
    }
    .signature-area {
      margin-top: 30px;
      font-size: 14px;
    }
    .small-text {
      font-size: 12px;
      color: #555;
      margin-top: 10px;
    }
    .amount-words {
      font-size: 15px;
      margin-top: 8px;
      font-style: italic;
      padding: 0 5px;
    }
    .qr {
      margin-top: 10px;
    }
    strong {
      font-weight: bold;
    }
    @media print {
      body {
        zoom: 1.7;
      }
    }
  </style>
  <script src="https://cdn.jsdelivr.net/npm/qrious/dist/qrious.min.js"></script>
</head>

<body onload="handlePrint()">

<div class="container">
  <div class="logo">
    <img src="../../assets/images/logo.png" alt="Company Logo">
  </div>

  <h2>PETTY CASH VOUCHER</h2>

  <div class="section">
    <div><strong>Branch:</strong> <?= htmlspecialchars($transaction['branch_name']) ?></div>
    <div><strong>Reference:</strong> <?= htmlspecialchars($transaction['reference_no']) ?></div>
    <div><strong>Date:</strong> <?= htmlspecialchars($transaction['transaction_date']) ?></div>
  </div>

  <div class="line"></div>

  <div class="section">
    <div><strong>Type:</strong> <?= ucfirst(htmlspecialchars($transaction['type'])) ?></div>
    <div><strong>Supplier:</strong> <?= htmlspecialchars($transaction['supplier_name'] ?? '-') ?></div>
    <div><strong>Payee:</strong> <?= htmlspecialchars($transaction['payee_name']) ?></div>
  </div>

  <div class="description">
    <strong>Description:</strong><br><?= htmlspecialchars($transaction['description']) ?>
  </div>

  <div class="section">
    <div><strong>Amount:</strong> KSh <?= number_format($transaction['amount'], 2) ?></div>
    <div><strong>Collected By:</strong> <?= htmlspecialchars($transaction['collected_by']) ?></div>
  </div>

  <div class="amount-words">
    <strong>In Words:</strong><br><?= strtoupper($amount_in_words) ?>
  </div>

  <div class="line"></div>

  <div class="signature-area">
    ___________________________<br>
    Signature of Payee
  </div>

  <div class="signature-area">
    ___________________________<br>
    Approved By
  </div>

  <div class="signature-area" style="margin-top:20px;">
    <strong>[ Company Stamp Here ]</strong>
  </div>

  <div class="qr">
    <canvas id="qr-code"></canvas>
  </div>

  <div class="small-text">Printed by SmartPay360</div>
</div>

<script>
new QRious({
  element: document.getElementById('qr-code'),
  value: "<?= htmlspecialchars($transaction['reference_no']) ?>",
  size: 80,
  level: 'H'
});

function handlePrint() {
  window.print();

  // Optional: redirect to summary page after print
  setTimeout(function () {
    window.location.href = 'list.php';
  }, 1500);
}
</script>

</body>
</html>
