<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// Validate payment ID
$payment_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($payment_id <= 0) {
    $_SESSION['error'] = "Invalid payment ID.";
    header("Location: list.php");
    exit;
}

// Fetch payment
$stmt = $conn->prepare("
    SELECT p.*, s.name AS supplier_name,
           (SELECT COUNT(*) FROM payment_allocations pa WHERE pa.payment_id = p.id) AS allocation_count,
           (SELECT SUM(pa.amount_allocated) FROM payment_allocations pa WHERE pa.payment_id = p.id) AS allocated_amount
    FROM payments p
    JOIN suppliers s ON p.supplier_id = s.id
    WHERE p.id = ?
");
$stmt->bind_param("i", $payment_id);
$stmt->execute();
$result = $stmt->get_result();
$payment = $result->fetch_assoc();

if (!$payment) {
    $_SESSION['error'] = "Payment not found.";
    header("Location: list.php");
    exit;
}

// Fetch allocations
$alloc_stmt = $conn->prepare("
    SELECT pa.invoice_id, i.invoice_number, b.branch_name, s.name AS supplier_name,
           pa.amount_allocated, i.total_payable,
           IFNULL((SELECT SUM(pa2.amount_allocated) FROM payment_allocations pa2 WHERE pa2.invoice_id = i.id), 0) AS total_paid,
           IFNULL((SELECT SUM(cn.amount) FROM credit_notes cn WHERE cn.invoice_id = i.id), 0) AS total_credit
    FROM payment_allocations pa
    JOIN invoices i ON pa.invoice_id = i.id
    JOIN branches b ON i.branch_id = b.id
    JOIN suppliers s ON i.supplier_id = s.id
    WHERE pa.payment_id = ?
    ORDER BY i.invoice_number
");
$alloc_stmt->bind_param("i", $payment_id);
$alloc_stmt->execute();
$alloc_result = $alloc_stmt->get_result();
$invoices = [];
while ($alloc = $alloc_result->fetch_assoc()) {
    $invoices[] = $alloc;
}

// Fetch payment logs
$log_stmt = $conn->prepare("
    SELECT l.*, u.full_name 
    FROM payment_logs l
    JOIN users u ON l.user_id = u.id
    WHERE l.payment_id = ?
    ORDER BY l.timestamp ASC
");
$log_stmt->bind_param("i", $payment_id);
$log_stmt->execute();
$log_result = $log_stmt->get_result();
$logs = [];
while ($log = $log_result->fetch_assoc()) {
    $logs[] = $log;
}
?>

<div class="main-content">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="fw-bold text-primary"><i class="fas fa-money-check-alt me-2"></i> Payment Details</h4>
        <div>
            <a href="receipt.php?id=<?= $payment_id ?>" class="btn btn-outline-secondary me-2">
                <i class="fas fa-file-pdf me-1"></i> View Receipt
            </a>
            <a href="list.php" class="btn btn-outline-primary">
                <i class="fas fa-arrow-left me-1"></i> Back to Payments
            </a>
        </div>
    </div>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($_SESSION['error']); unset($_SESSION['error']); ?></div>
    <?php endif; ?>

    <!-- Payment Summary -->
    <div class="card shadow-sm border-0 mb-4">
        <div class="card-body">
            <h5 class="card-title text-primary">Payment #<?= $payment['id'] ?> - <?= htmlspecialchars($payment['reference_no']) ?></h5>
            <div class="row g-3">
                <div class="col-md-6">
                    <p><strong>Supplier:</strong> <?= htmlspecialchars($payment['supplier_name']) ?></p>
                    <p><strong>Amount:</strong> KSh <?= number_format($payment['amount'], 2) ?></p>
                    <p><strong>Allocated:</strong> KSh <?= number_format($payment['allocated_amount'] ?? 0, 2) ?></p>
                    <p><strong>Unallocated:</strong> KSh <?= number_format($payment['amount'] - ($payment['allocated_amount'] ?? 0), 2) ?></p>
                </div>
                <div class="col-md-6">
                    <p><strong>Payment Date:</strong> <?= date('d-M-Y', strtotime($payment['payment_date'])) ?></p>
                    <p><strong>Method:</strong> <?= htmlspecialchars($payment['method']) ?></p>
                    <p><strong>Reference No:</strong> <?= htmlspecialchars($payment['reference_no']) ?></p>
                    <p><strong>Remarks:</strong> <?= htmlspecialchars($payment['remarks'] ?: 'N/A') ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Invoice Allocations -->
    <h5 class="text-primary mb-3"><i class="fas fa-list me-2"></i> Invoice Allocations (<?= $payment['allocation_count'] ?>)</h5>
    <div class="card shadow-sm border-0">
        <div class="card-body table-responsive">
            <table class="table table-hover table-bordered align-middle small">
                <thead class="table-dark">
                    <tr class="text-center">
                        <th>#</th>
                        <th>Invoice No</th>
                        <th>Supplier</th>
                        <th>Branch</th>
                        <th>Total Payable</th>
                        <th>Paid</th>
                        <th>Credit</th>
                        <th>Balance</th>
                        <th>Allocated</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($invoices): foreach ($invoices as $i => $a): ?>
                        <?php $balance = $a['total_payable'] - $a['total_paid'] - $a['total_credit']; ?>
                        <tr>
                            <td class="text-center"><?= $i + 1 ?></td>
                            <td><a href="../invoices/view.php?id=<?= $a['invoice_id'] ?>" class="text-primary"><?= htmlspecialchars($a['invoice_number']) ?></a></td>
                            <td><?= htmlspecialchars($a['supplier_name']) ?></td>
                            <td><?= htmlspecialchars($a['branch_name']) ?></td>
                            <td class="text-end">KSh <?= number_format($a['total_payable'], 2) ?></td>
                            <td class="text-end text-muted">KSh <?= number_format($a['total_paid'], 2) ?></td>
                            <td class="text-end text-warning">KSh <?= number_format($a['total_credit'], 2) ?></td>
                            <td class="text-end text-success fw-bold">KSh <?= number_format($balance, 2) ?></td>
                            <td class="text-end text-primary">KSh <?= number_format($a['amount_allocated'], 2) ?></td>
                        </tr>
                    <?php endforeach; else: ?>
                        <tr><td colspan="9" class="text-center text-muted">No invoices allocated.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Payment Logs -->
    <h5 class="text-primary mt-5 mb-3"><i class="fas fa-history me-2"></i>Payment Activity Log</h5>
    <div class="card shadow-sm border-0">
        <div class="card-body table-responsive">
            <table class="table table-bordered table-sm small">
                <thead class="table-dark">
                    <tr class="text-center">
                        <th>#</th>
                        <th>Action</th>
                        <th>Performed By</th>
                        <th>Remarks</th>
                        <th>Date & Time</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($logs): foreach ($logs as $i => $log): ?>
                        <tr>
                            <td class="text-center"><?= $i + 1 ?></td>
                            <td class="text-center"><?= htmlspecialchars($log['action']) ?></td>
                            <td><?= htmlspecialchars($log['full_name']) ?></td>
                            <td><?= htmlspecialchars($log['remarks'] ?: '-') ?></td>
                            <td class="text-center"><?= date('d M Y, h:i A', strtotime($log['timestamp'])) ?></td>
                        </tr>
                    <?php endforeach; else: ?>
                        <tr><td colspan="5" class="text-center text-muted">No log entries available.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php include '../../includes/footer.php'; ?>
