<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

$supplier_id = $_GET['supplier_id'] ?? 0;
$branch_id = $_GET['branch_id'] ?? 0;
$from_date = $_GET['from_date'] ?? '';
$to_date = $_GET['to_date'] ?? '';

// Validate supplier
$valid_supplier = false;
if ($supplier_id > 0) {
    $stmt = $conn->prepare("SELECT id FROM suppliers WHERE id = ?");
    $stmt->bind_param("i", $supplier_id);
    $stmt->execute();
    $valid_supplier = $stmt->get_result()->num_rows > 0;
    $stmt->close();
    if (!$valid_supplier) {
        $_SESSION['error'] = "Invalid supplier.";
        $supplier_id = 0;
    }
}

$conditions = [];
$params = [];
$types = '';

if ($supplier_id > 0) {
    $conditions[] = "t.supplier_id = ?";
    $params[] = $supplier_id;
    $types .= 'i';
}
if ($branch_id > 0) {
    $conditions[] = "(t.transaction_type = 'invoice' AND t.branch_id = ?)";
    $params[] = $branch_id;
    $types .= 'i';
}
if (!empty($from_date)) {
    $conditions[] = "t.transaction_date >= ?";
    $params[] = $from_date;
    $types .= 's';
}
if (!empty($to_date)) {
    $conditions[] = "t.transaction_date <= ?";
    $params[] = $to_date;
    $types .= 's';
}
$where = $conditions ? 'WHERE ' . implode(' AND ', $conditions) : '';

// Fetch transactions
$query = "
SELECT 
    t.transaction_date, t.description, t.debit, t.credit, 0 as balance,
    t.transaction_type, t.reference_id, t.supplier_name, t.branch_name, t.branch_id, t.supplier_id
FROM (
    SELECT i.invoice_date AS transaction_date, CONCAT('Invoice #', i.invoice_number) AS description, 
           i.total_amount AS debit, 0 AS credit, 'invoice' AS transaction_type,
           i.id AS reference_id, s.id AS supplier_id, s.name AS supplier_name, 
           COALESCE(b.branch_name, 'N/A') AS branch_name, i.branch_id
    FROM invoices i
    JOIN suppliers s ON i.supplier_id = s.id
    LEFT JOIN branches b ON i.branch_id = b.id

    UNION ALL

    SELECT p.payment_date, CONCAT('Payment - Ref: ', p.reference_no), 
           0, p.amount, 'payment', p.id, s.id, s.name,
           (SELECT branch_name FROM payment_allocations pa JOIN invoices i2 ON pa.invoice_id = i2.id LEFT JOIN branches b2 ON i2.branch_id = b2.id WHERE pa.payment_id = p.id LIMIT 1),
           NULL
    FROM payments p
    JOIN suppliers s ON p.supplier_id = s.id

    UNION ALL

    SELECT cn.credit_date, CONCAT('Credit Note #', cn.credit_note_number), 
           0, cn.amount, 'credit_note', cn.id, s.id, s.name,
           COALESCE(b.branch_name, 'N/A'), i.branch_id
    FROM credit_notes cn
    JOIN suppliers s ON cn.supplier_id = s.id
    LEFT JOIN invoices i ON cn.invoice_id = i.id
    LEFT JOIN branches b ON i.branch_id = b.id
) t
$where
ORDER BY t.transaction_date ASC, t.transaction_type DESC
";

$stmt = $conn->prepare($query);
if ($params) $stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();

$transactions = [];
$running_balance = 0;
$supplier_name = '';
foreach ($result as $row) {
    $running_balance += $row['debit'] - $row['credit'];
    $row['balance'] = $running_balance;
    $transactions[] = $row;
    $supplier_name = $row['supplier_name'];
}
$stmt->close();

// Fetch related payments and credit notes per invoice
$invoice_ids = array_column(array_filter($transactions, fn($t) => $t['transaction_type'] === 'invoice'), 'reference_id');
$invoice_links = [];

if (!empty($invoice_ids)) {
    $ids = implode(',', array_map('intval', $invoice_ids));

    $res = $conn->query("
        SELECT pa.invoice_id, p.reference_no, p.payment_date, pa.allocated_amount
        FROM payment_allocations pa
        JOIN payments p ON pa.payment_id = p.id
        WHERE pa.invoice_id IN ($ids)
    ");
    if ($res) {
        while ($row = $res->fetch_assoc()) {
            $invoice_links[$row['invoice_id']]['payments'][] = $row;
        }
    }

    $res = $conn->query("
        SELECT invoice_id, credit_note_number, credit_date, amount
        FROM credit_notes
        WHERE invoice_id IN ($ids)
    ");
    if ($res) {
        while ($row = $res->fetch_assoc()) {
            $invoice_links[$row['invoice_id']]['credit_notes'][] = $row;
        }
    }
}

// Totals
$total_query = "
SELECT 
    SUM(CASE WHEN t.transaction_type = 'invoice' THEN t.debit ELSE 0 END) AS total_invoiced,
    SUM(CASE WHEN t.transaction_type = 'payment' THEN t.credit ELSE 0 END) AS total_paid,
    SUM(CASE WHEN t.transaction_type = 'credit_note' THEN t.credit ELSE 0 END) AS total_credit_notes
FROM (
    SELECT 'invoice' AS transaction_type, i.total_amount AS debit, 0 AS credit, s.id AS supplier_id, i.branch_id AS branch_id
    FROM invoices i JOIN suppliers s ON i.supplier_id = s.id
    UNION ALL
    SELECT 'payment', 0, p.amount, s.id, NULL FROM payments p JOIN suppliers s ON p.supplier_id = s.id
    UNION ALL
    SELECT 'credit_note', 0, cn.amount, s.id, i.branch_id FROM credit_notes cn JOIN suppliers s ON cn.supplier_id = s.id LEFT JOIN invoices i ON cn.invoice_id = i.id
) t
$where
";

$total_stmt = $conn->prepare($total_query);
if ($params) $total_stmt->bind_param($types, ...$params);
$total_stmt->execute();
$summary = $total_stmt->get_result()->fetch_assoc();
$total_stmt->close();

$total_invoiced = $summary['total_invoiced'] ?? 0;
$total_paid = $summary['total_paid'] ?? 0;
$total_credit_notes = $summary['total_credit_notes'] ?? 0;
$balance_due = $total_invoiced - $total_paid - $total_credit_notes;

// Dropdowns
$suppliers = $conn->query("SELECT id, name FROM suppliers ORDER BY name");
$branches = $conn->query("SELECT id, branch_name FROM branches WHERE status='active' ORDER BY branch_name");
?>

<!DOCTYPE html>
<html>
<head>
    <title>Supplier Statement</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .debit-row { background-color: #f8d7da; }
        .credit-row { background-color: #d1e7dd; }
        .info-row   { background-color: #cff4fc; }
        .expand-row { background-color: #fefefe; }
    </style>
</head>
<body class="p-4">
<div class="container">
    <h4 class="fw-bold text-primary mb-3">📄 Supplier Statement</h4>

    <form method="GET" class="row g-2 mb-4">
        <div class="col-md-3">
            <select name="supplier_id" class="form-select" required>
                <option value="">-- Select Supplier --</option>
                <?php while ($s = $suppliers->fetch_assoc()): ?>
                    <option value="<?= $s['id'] ?>" <?= $supplier_id == $s['id'] ? 'selected' : '' ?>><?= htmlspecialchars($s['name']) ?></option>
                <?php endwhile; ?>
            </select>
        </div>
        <div class="col-md-3">
            <select name="branch_id" class="form-select">
                <option value="">-- All Branches --</option>
                <?php while ($b = $branches->fetch_assoc()): ?>
                    <option value="<?= $b['id'] ?>" <?= $branch_id == $b['id'] ? 'selected' : '' ?>><?= htmlspecialchars($b['branch_name']) ?></option>
                <?php endwhile; ?>
            </select>
        </div>
        <div class="col-md-2"><input type="date" name="from_date" value="<?= $from_date ?>" class="form-control"></div>
        <div class="col-md-2"><input type="date" name="to_date" value="<?= $to_date ?>" class="form-control"></div>
        <div class="col-md-2"><button class="btn btn-primary w-100">Filter</button></div>
    </form>

    <div class="alert alert-info fw-bold">
        <div class="row">
            <div class="col">💸 Invoiced: KSh <?= number_format($total_invoiced, 2) ?></div>
            <div class="col">💰 Paid: KSh <?= number_format($total_paid, 2) ?></div>
            <div class="col">🎟 Credit Notes: KSh <?= number_format($total_credit_notes, 2) ?></div>
            <div class="col">⚖ Balance: <span class="text-<?= $balance_due > 0 ? 'danger' : 'success' ?>">KSh <?= number_format($balance_due, 2) ?></span></div>
        </div>
    </div>

    <table class="table table-bordered table-sm">
        <thead class="table-dark text-center">
        <tr>
            <th>Date</th><th>Supplier</th><th>Branch</th><th>Description</th>
            <th>Debit</th><th>Credit</th><th>Balance</th>
        </tr>
        </thead>
        <tbody class="text-center">
        <?php foreach ($transactions as $row): ?>
            <tr class="<?= $row['transaction_type'] === 'invoice' ? 'debit-row' : ($row['transaction_type'] === 'payment' ? 'credit-row' : 'info-row') ?>"
                <?= $row['transaction_type'] === 'invoice' && isset($invoice_links[$row['reference_id']]) ? "style='cursor:pointer'" : "" ?>
                onclick="document.getElementById('expand-<?= $row['reference_id'] ?>')?.classList.toggle('d-none')">
                <td><?= date('d-M-Y', strtotime($row['transaction_date'])) ?></td>
                <td><?= htmlspecialchars($row['supplier_name']) ?></td>
                <td><?= htmlspecialchars($row['branch_name']) ?></td>
                <td><?= htmlspecialchars($row['description']) ?></td>
                <td class="text-end"><?= $row['debit'] > 0 ? number_format($row['debit'], 2) : '-' ?></td>
                <td class="text-end"><?= $row['credit'] > 0 ? number_format($row['credit'], 2) : '-' ?></td>
                <td class="text-end fw-bold"><?= number_format($row['balance'], 2) ?></td>
            </tr>
            <?php if ($row['transaction_type'] === 'invoice' && isset($invoice_links[$row['reference_id']])): ?>
                <tr class="expand-row d-none" id="expand-<?= $row['reference_id'] ?>">
                    <td colspan="7" class="text-start">
                        <b>→ Related Payments:</b><br>
                        <?php foreach ($invoice_links[$row['reference_id']]['payments'] ?? [] as $p): ?>
                            • <?= date('d-M-Y', strtotime($p['payment_date'])) ?> — Ref: <?= $p['reference_no'] ?> — Amount: KSh <?= number_format($p['allocated_amount'], 2) ?><br>
                        <?php endforeach; ?>
                        <b>→ Related Credit Notes:</b><br>
                        <?php foreach ($invoice_links[$row['reference_id']]['credit_notes'] ?? [] as $c): ?>
                            • <?= date('d-M-Y', strtotime($c['credit_date'])) ?> — #<?= $c['credit_note_number'] ?> — Amount: KSh <?= number_format($c['amount'], 2) ?><br>
                        <?php endforeach; ?>
                    </td>
                </tr>
            <?php endif; ?>
        <?php endforeach; ?>
        </tbody>
    </table>
</div>
</body>
</html>
