<?php
require_once '../../config/db.php';

$date = $_POST['date'] ?? '';
if (!$date) {
    echo '<p class="text-danger">Invalid request: Date is missing.</p>';
    error_log("fetch_transactions.php: Date missing.");
    exit;
}

// Log request parameters for debugging
error_log("fetch_transactions.php: date=$date");

// Query for invoices
$invoice_query = "
    SELECT 
        i.invoice_date AS transaction_date,
        CONCAT('Invoice #', i.invoice_number) AS description,
        i.total_amount AS amount,
        'invoice' AS transaction_type,
        i.id AS reference_id,
        COALESCE(s.name, 'Unknown') AS supplier_name,
        COALESCE(b.branch_name, 'N/A') AS branch_name
    FROM invoices i
    LEFT JOIN suppliers s ON i.supplier_id = s.id
    LEFT JOIN branches b ON i.branch_id = b.id
    WHERE DATE(i.invoice_date) = ?
    ORDER BY i.invoice_date ASC
";

$invoice_stmt = $conn->prepare($invoice_query);
if ($invoice_stmt === false) {
    echo '<p class="text-danger">Invoice query preparation failed: ' . htmlspecialchars($conn->error) . '</p>';
    error_log("fetch_transactions.php: Invoice query preparation failed: " . $conn->error);
    exit;
}
$invoice_stmt->bind_param('s', $date);
if (!$invoice_stmt->execute()) {
    echo '<p class="text-danger">Invoice query execution failed: ' . htmlspecialchars($invoice_stmt->error) . '</p>';
    error_log("fetch_transactions.php: Invoice query execution failed: " . $invoice_stmt->error);
    exit;
}
$invoice_result = $invoice_stmt->get_result();
$invoices = [];
while ($row = $invoice_result->fetch_assoc()) {
    $invoices[] = $row;
}
$invoice_stmt->close();

// Query for payments
$payment_query = "
    SELECT 
        p.payment_date AS transaction_date,
        CONCAT('Payment - Invoices: ', COALESCE((
            SELECT GROUP_CONCAT(i.invoice_number ORDER BY i.invoice_number SEPARATOR ', ') 
            FROM payment_allocations pa 
            JOIN invoices i ON pa.invoice_id = i.id 
            WHERE pa.payment_id = p.id
        ), 'None')) AS description,
        p.amount AS amount,
        'payment' AS transaction_type,
        p.id AS reference_id,
        COALESCE(s.name, 'Unknown') AS supplier_name,
        COALESCE((
            SELECT b2.branch_name 
            FROM payment_allocations pa 
            JOIN invoices i2 ON pa.invoice_id = i2.id 
            LEFT JOIN branches b2 ON i2.branch_id = b2.id 
            WHERE pa.payment_id = p.id 
            LIMIT 1
        ), 'N/A') AS branch_name
    FROM payments p
    LEFT JOIN suppliers s ON p.supplier_id = s.id
    WHERE DATE(p.payment_date) = ?
    ORDER BY p.payment_date ASC
";

$payment_stmt = $conn->prepare($payment_query);
if ($payment_stmt === false) {
    echo '<p class="text-danger">Payment query preparation failed: ' . htmlspecialchars($conn->error) . '</p>';
    error_log("fetch_transactions.php: Payment query preparation failed: " . $conn->error);
    exit;
}
$payment_stmt->bind_param('s', $date);
if (!$payment_stmt->execute()) {
    echo '<p class="text-danger">Payment query execution failed: ' . htmlspecialchars($payment_stmt->error) . '</p>';
    error_log("fetch_transactions.php: Payment query execution failed: " . $payment_stmt->error);
    exit;
}
$payment_result = $payment_stmt->get_result();
$payments = [];
$total_payments = 0;
while ($row = $payment_result->fetch_assoc()) {
    $payments[] = $row;
    $total_payments += $row['amount'];
}
$payment_stmt->close();

if (empty($invoices) && empty($payments)) {
    echo '<p class="text-muted">No transactions found for this date.</p>';
    exit;
}
?>

<style>
    .transaction-table { width: 100%; table-layout: auto; margin-bottom: 20px; }
    .transaction-table th, .transaction-table td { 
        white-space: nowrap; 
        padding: 8px; 
        text-align: left; 
        vertical-align: middle; 
    }
    .transaction-table th { font-weight: bold; }
    .transaction-table .text-end { text-align: right; }
    .transaction-table.invoices-table thead { background-color: #ff9999; }
    .transaction-table.payments-table thead { background-color: #99cc99; }
    .transaction-table.invoices-table tbody tr { background-color: #ffe6e6; }
    .transaction-table.payments-table tbody tr { background-color: #e6ffe6; }
    .transaction-table tfoot td { font-weight: bold; background-color: #f8f9fa; }
    h6 { margin-top: 20px; margin-bottom: 10px; }
</style>

<?php if (!empty($invoices)): ?>
    <h6>Invoices</h6>
    <table class="table table-sm table-bordered transaction-table invoices-table" aria-label="Invoices for <?= htmlspecialchars($date) ?>">
        <thead>
            <tr>
                <th>Date</th>
                <th>Supplier</th>
                <th>Branch</th>
                <th>Description</th>
                <th>Amount</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($invoices as $row): ?>
                <tr>
                    <td><?= date('d-M-Y H:i', strtotime($row['transaction_date'])) ?></td>
                    <td><?= htmlspecialchars($row['supplier_name']) ?></td>
                    <td><?= htmlspecialchars($row['branch_name']) ?></td>
                    <td><?= htmlspecialchars($row['description']) ?></td>
                    <td class="text-end">KSh <?= number_format($row['amount'], 2) ?></td>
                    <td>
                        <a href="view_invoice.php?id=<?= $row['reference_id'] ?>" class="btn btn-sm btn-outline-primary">View</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>

<?php if (!empty($payments)): ?>
    <h6>Payments</h6>
    <table class="table table-sm table-bordered transaction-table payments-table" aria-label="Payments for <?= htmlspecialchars($date) ?>">
        <thead>
            <tr>
                <th>Date</th>
                <th>Supplier</th>
                <th>Branch</th>
                <th>Description</th>
                <th>Amount</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($payments as $row): ?>
                <tr>
                    <td><?= date('d-M-Y H:i', strtotime($row['transaction_date'])) ?></td>
                    <td><?= htmlspecialchars($row['supplier_name']) ?></td>
                    <td><?= htmlspecialchars($row['branch_name']) ?></td>
                    <td><?= htmlspecialchars($row['description']) ?></td>
                    <td class="text-end">KSh <?= number_format($row['amount'], 2) ?></td>
                    <td>
                        <a href="view_payment.php?id=<?= $row['reference_id'] ?>" class="btn btn-sm btn-outline-primary">View</a>
                        <a href="receipt.php?id=<?= $row['reference_id'] ?>" class="btn btn-sm btn-outline-secondary">PDF</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr>
                <td colspan="4"></td>
                <td class="text-end">Total Payments: KSh <?= number_format($total_payments, 2) ?></td>
                <td></td>
            </tr>
        </tfoot>
    </table>
<?php endif; ?>