<?php
require_once '../../config/db.php';

// Set content type to JSON
header('Content-Type: application/json');

// Build payments query
$supplier_id = isset($_GET['supplier_id']) ? (int)$_GET['supplier_id'] : null;
$query = "
  SELECT p.id, p.supplier_id, s.name AS supplier_name, p.amount, 
         DATE_FORMAT(p.payment_date, '%d-%b-%Y') AS payment_date, p.created_at,
         (SELECT GROUP_CONCAT(i.invoice_number ORDER BY i.invoice_number SEPARATOR ', ') 
          FROM payment_allocations pa 
          JOIN invoices i ON pa.invoice_id = i.id 
          WHERE pa.payment_id = p.id) AS invoice_numbers
  FROM payments p
  JOIN suppliers s ON p.supplier_id = s.id
";
$params = [];
if ($supplier_id !== null) {
  $query .= " WHERE p.supplier_id = ?";
  $params[] = $supplier_id;
}
$query .= " ORDER BY p.payment_date DESC, p.created_at DESC";

// Execute payments query
$stmt = $conn->prepare($query);
if (!$stmt) {
  error_log("Prepare failed: " . $conn->error);
  echo json_encode(['error' => 'Database error']);
  exit;
}
if ($supplier_id !== null) {
  $stmt->bind_param("i", $supplier_id);
}
$stmt->execute();
$result = $stmt->get_result();

$payments = [];
while ($row = $result->fetch_assoc()) {
  $row['invoice_numbers'] = $row['invoice_numbers'] ?: 'None';
  $payments[] = $row;
}

// Build summary query for unpaid invoices
$summary_query = "
  SELECT s.name AS supplier_name,
         COUNT(i.id) AS unpaid_invoice_count,
         IFNULL(SUM(i.total_payable - IFNULL((SELECT SUM(pa.amount_allocated) FROM payment_allocations pa WHERE pa.invoice_id = i.id), 0) - IFNULL((SELECT SUM(cn.amount) FROM credit_notes cn WHERE cn.invoice_id = i.id), 0)), 0) AS total_unpaid_balance
  FROM invoices i
  JOIN suppliers s ON i.supplier_id = s.id
  WHERE i.supplier_id = ? 
  AND (i.total_payable - IFNULL((SELECT SUM(pa.amount_allocated) FROM payment_allocations pa WHERE pa.invoice_id = i.id), 0) - IFNULL((SELECT SUM(cn.amount) FROM credit_notes cn WHERE cn.invoice_id = i.id), 0)) > 0
";
$invoices_query = "
  SELECT i.invoice_number, i.total_payable,
         IFNULL((SELECT SUM(pa.amount_allocated) FROM payment_allocations pa WHERE pa.invoice_id = i.id), 0) AS paid,
         (i.total_payable - IFNULL((SELECT SUM(pa.amount_allocated) FROM payment_allocations pa WHERE pa.invoice_id = i.id), 0) - IFNULL((SELECT SUM(cn.amount) FROM credit_notes cn WHERE cn.invoice_id = i.id), 0)) AS balance
  FROM invoices i
  WHERE i.supplier_id = ?
  AND (i.total_payable - IFNULL((SELECT SUM(pa.amount_allocated) FROM payment_allocations pa WHERE pa.invoice_id = i.id), 0) - IFNULL((SELECT SUM(cn.amount) FROM credit_notes cn WHERE cn.invoice_id = i.id), 0)) > 0
  ORDER BY i.invoice_number
";

$summary = [
  'supplier_name' => 'All Suppliers',
  'unpaid_invoice_count' => 0,
  'total_unpaid_balance' => 0,
  'unpaid_invoices' => []
];

if ($supplier_id !== null) {
  // Execute summary query
  $summary_stmt = $conn->prepare($summary_query);
  if (!$summary_stmt) {
    error_log("Summary prepare failed: " . $conn->error);
    echo json_encode(['error' => 'Database error']);
    exit;
  }
  $summary_stmt->bind_param("i", $supplier_id);
  $summary_stmt->execute();
  $summary_result = $summary_stmt->get_result();
  $summary_data = $summary_result->fetch_assoc();
  if ($summary_data) {
    $summary['supplier_name'] = $summary_data['supplier_name'];
    $summary['unpaid_invoice_count'] = $summary_data['unpaid_invoice_count'];
    $summary['total_unpaid_balance'] = $summary_data['total_unpaid_balance'];
  }

  // Execute invoices query
  $invoices_stmt = $conn->prepare($invoices_query);
  if (!$invoices_stmt) {
    error_log("Invoices prepare failed: " . $conn->error);
    echo json_encode(['error' => 'Database error']);
    exit;
  }
  $invoices_stmt->bind_param("i", $supplier_id);
  $invoices_stmt->execute();
  $invoices_result = $invoices_stmt->get_result();
  $unpaid_invoices = [];
  while ($row = $invoices_result->fetch_assoc()) {
    $unpaid_invoices[] = $row;
  }
  $summary['unpaid_invoices'] = $unpaid_invoices;
}

// Output JSON
echo json_encode([
  'payments' => $payments,
  'summary' => $summary
]);
exit;