<?php
session_start();
require_once '../../config/db.php';

$vehicles = $conn->query("SELECT * FROM vehicles ORDER BY plate_number");
if (!$vehicles) {
    die("Vehicles Query Failed: " . $conn->error);
}
$vehicles = $vehicles->fetch_all(MYSQLI_ASSOC);

$routes = $conn->query("SELECT * FROM routes ORDER BY name");
if (!$routes) {
    die("Routes Query Failed: " . $conn->error);
}
$routes = $routes->fetch_all(MYSQLI_ASSOC);

$skus = $conn->query("SELECT * FROM skus ORDER BY id");
if (!$skus) {
    die("SKUs Query Failed: " . $conn->error);
}
$skus = $skus->fetch_all(MYSQLI_ASSOC);

$created_by = $_SESSION['user_name'] ?? 'admin';

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $vehicle_id = $_POST['vehicle_id'];
    $route_id = $_POST['route_id'];
    $loading_date = $_POST['loading_date'];
    $mileage_start = $_POST['mileage_start'];
    $remarks = $_POST['remarks'] ?? '';

    $stmt = $conn->prepare("INSERT INTO loadings (vehicle_id, route_id, loading_date, mileage_start, created_by, remarks) VALUES (?, ?, ?, ?, ?, ?)");
    if (!$stmt) {
        die("Insert Error: " . $conn->error);
    }
    $stmt->bind_param('iissss', $vehicle_id, $route_id, $loading_date, $mileage_start, $created_by, $remarks);
    $stmt->execute();
    $loading_id = $conn->insert_id;

    foreach ($_POST['crates'] as $sku_id => $crates) {
        $crates = (int)$crates;
        if ($crates > 0) {
            $stmt_item = $conn->prepare("INSERT INTO loading_items (loading_id, sku_id, crates_loaded) VALUES (?, ?, ?)");
            if ($stmt_item) {
                $stmt_item->bind_param('iii', $loading_id, $sku_id, $crates);
                $stmt_item->execute();
            }
        }
    }

    header("Location: loading_sheet.php?id=" . $loading_id);
    exit;
}

include '../../includes/layout2.php';
?>

<div class="main-content container py-4">
    <h4 class="fw-bold text-primary mb-4"><i class="fas fa-bread-slice me-2"></i> Load Bread for Delivery</h4>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error']; unset($_SESSION['error']); ?></div>
    <?php endif; ?>

    <div class="card shadow-sm border-0">
        <div class="card-body">
            <form method="POST" id="loadingForm">

                <div class="row mb-4 g-3">
                    <div class="col-md-4">
                        <label class="form-label">Vehicle <span class="text-danger">*</span></label>
                        <select name="vehicle_id" id="vehicle_id" class="form-select" required>
                            <option value="">Select Vehicle</option>
                            <?php foreach ($vehicles as $v): ?>
                                <option value="<?= $v['id'] ?>" data-max="<?= $v['max_crates'] ?? 0 ?>">
                                    <?= htmlspecialchars($v['plate_number']) ?> - <?= htmlspecialchars($v['driver_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Route <span class="text-danger">*</span></label>
                        <select name="route_id" id="route_id" class="form-select" required>
                            <option value="">Select Route</option>
                            <?php foreach ($routes as $r): ?>
                                <option value="<?= $r['id'] ?>"><?= htmlspecialchars($r['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Loading Date <span class="text-danger">*</span></label>
                        <input type="date" name="loading_date" class="form-control" value="<?= date('Y-m-d') ?>" required>
                    </div>
                </div>

                <div class="row mb-4 g-3">
                    <div class="col-md-4">
                        <label class="form-label">Mileage (Start) <span class="text-danger">*</span></label>
                        <input type="number" name="mileage_start" class="form-control" required>
                    </div>
                    <div class="col-md-8">
                        <label class="form-label">Remarks (Optional)</label>
                        <textarea name="remarks" class="form-control" rows="2" placeholder="E.g. Fuel low, delays..."></textarea>
                    </div>
                </div>

                <div class="mb-3">
                    <button type="button" class="btn btn-outline-secondary btn-sm" onclick="loadPrevious()">
                        📋 Load Same as Last Time
                    </button>
                </div>

                <h5 class="mb-3">Bread Loaded:</h5>

                <div class="table-responsive">
                    <table class="table table-bordered table-sm align-middle">
                        <thead class="table-light">
                            <tr class="text-center">
                                <th>Bread SKU</th>
                                <th>Crates</th>
                                <th>Price/Crate</th>
                                <th>Expected Value</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($skus as $sku): ?>
                                <tr>
                                    <td><?= htmlspecialchars($sku['name']) ?></td>
                                    <td class="text-center">
                                        <input type="number" name="crates[<?= $sku['id'] ?>]" class="form-control crates text-center" min="0" value="0">
                                    </td>
                                    <td class="text-end">Ksh <?= number_format($sku['price_per_crate'], 2) ?></td>
                                    <td class="expected-val text-end" data-price="<?= $sku['price_per_crate'] ?>">0.00</td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="sticky-summary d-flex justify-content-between fw-bold mt-3 p-2 border-top">
                    <div>Total Crates: <span id="total-crates">0</span></div>
                    <div>Total Expected: Ksh <span id="grand-total">0.00</span></div>
                </div>

                <div class="mt-4 text-end">
                    <button type="submit" class="btn btn-primary px-4">
                        ✅ Save & Print Loading Sheet
                    </button>
                    <a href="list.php" class="btn btn-secondary">Cancel</a>
                </div>

            </form>
        </div>
    </div>
</div>

<script>
const cratesInputs = document.querySelectorAll('.crates');
const expectedVals = document.querySelectorAll('.expected-val');
const grandTotal = document.getElementById('grand-total');
const totalCrates = document.getElementById('total-crates');
const vehicleSelect = document.getElementById('vehicle_id');

function calculate() {
    let total = 0, cratesSum = 0;
    cratesInputs.forEach((input, idx) => {
        let qty = parseInt(input.value) || 0;
        let price = parseFloat(expectedVals[idx].dataset.price);
        let val = qty * price;
        expectedVals[idx].textContent = val.toFixed(2);
        total += val;
        cratesSum += qty;
    });
    totalCrates.textContent = cratesSum;
    grandTotal.textContent = total.toFixed(2);
}

cratesInputs.forEach(input => input.addEventListener('input', calculate));
vehicleSelect.addEventListener('change', calculate);

function loadPrevious() {
    const routeId = document.getElementById('route_id').value;
    if (!routeId) return alert("⚠️ Please select a route first.");
    fetch(`ajax/get_last_loading.php?route_id=${routeId}`)
        .then(res => res.json())
        .then(data => {
            for (const [skuId, crates] of Object.entries(data)) {
                const input = document.querySelector(`input[name='crates[${skuId}]']`);
                if (input) input.value = crates;
            }
            calculate();
        })
        .catch(() => alert("Failed to load previous loading."));
}

window.onload = calculate;
</script>
