<?php
require_once '../../config/db.php';
include '../../includes/layout2.php';

$date = $_GET['date'] ?? date('Y-m-d');
$branch_id = $_GET['branch_id'] ?? 'all';

// Fetch branch list
$branches = [];
$result = $conn->query("SELECT id, branch_name FROM branches ORDER BY branch_name");
while ($row = $result->fetch_assoc()) {
    $branches[] = $row;
}

// Build dynamic query
$params = [$date];
$where = "a.attendance_date = ?";
if ($branch_id !== 'all') {
    $where .= " AND a.branch_id = ?";
    $params[] = $branch_id;
}

$stmt = $conn->prepare("
    SELECT 
        a.staff_id, s.first_name, s.last_name, s.national_id, a.attendance_date,
        a.status, a.in_time, a.out_time, b.branch_name 
    FROM attendance a
    JOIN staff s ON a.staff_id = s.staff_id
    JOIN branches b ON a.branch_id = b.id
    WHERE $where
    ORDER BY s.first_name, s.last_name
");

if (count($params) === 2) {
    $stmt->bind_param('si', ...$params);
} else {
    $stmt->bind_param('s', ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

$records = [];
$present = 0;
$absent = 0;
while ($row = $result->fetch_assoc()) {
    if ($row['status'] === 'present') $present++;
    elseif ($row['status'] === 'absent') $absent++;
    $records[] = $row;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Attendance Records</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.08); }
        .badge-success { background-color: #28a745; }
        .badge-danger { background-color: #dc3545; }
        .summary { font-size: 0.95rem; margin-top: 5px; }
    </style>
</head>
<body>
<div class="container mt-4">
    <h3 class="mb-4">📅 Attendance Records for <?= htmlspecialchars($date) ?></h3>

    <div class="card p-4 mb-4">
        <form method="GET" class="row g-3 align-items-end">
            <div class="col-md-4">
                <label for="date" class="form-label">Date</label>
                <input type="date" name="date" id="date" value="<?= htmlspecialchars($date) ?>" class="form-control" onchange="this.form.submit()">
            </div>
            <div class="col-md-4">
                <label for="branch_id" class="form-label">Branch</label>
                <select name="branch_id" id="branch_id" class="form-select" onchange="this.form.submit()">
                    <option value="all">All Branches</option>
                    <?php foreach ($branches as $b): ?>
                        <option value="<?= $b['id'] ?>" <?= ($branch_id == $b['id']) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($b['branch_name']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-4">
                <div class="summary text-muted">
                    <strong><?= count($records) ?></strong> record(s) —
                    <span class="text-success"><?= $present ?> present</span>,
                    <span class="text-danger"><?= $absent ?> absent</span>
                </div>
            </div>
        </form>
    </div>

    <?php if (count($records) > 0): ?>
        <div class="card p-3">
            <table class="table table-hover table-bordered">
                <thead class="table-light">
                    <tr>
                        <th>Staff ID</th>
                        <th>Name</th>
                        <th>National ID</th>
                        <th>Branch</th>
                        <th>Date</th>
                        <th>IN Time</th>
                        <th>OUT Time</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($records as $rec): ?>
                        <tr>
                            <td><?= htmlspecialchars($rec['staff_id']) ?></td>
                            <td><?= htmlspecialchars($rec['first_name'] . ' ' . $rec['last_name']) ?></td>
                            <td><?= htmlspecialchars($rec['national_id']) ?></td>
                            <td><?= htmlspecialchars($rec['branch_name']) ?></td>
                            <td><?= htmlspecialchars($rec['attendance_date']) ?></td>
                            <td><?= $rec['in_time'] ? date('H:i:s', strtotime($rec['in_time'])) : '-' ?></td>
                            <td><?= $rec['out_time'] ? date('H:i:s', strtotime($rec['out_time'])) : '-' ?></td>
                            <td>
                                <span class="badge <?= $rec['status'] == 'present' ? 'badge-success' : 'badge-danger' ?>">
                                    <?= ucfirst($rec['status']) ?>
                                </span>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php else: ?>
        <div class="alert alert-info">No attendance records found for this date<?= $branch_id !== 'all' ? ' and branch' : '' ?>.</div>
    <?php endif; ?>
</div>
</body>
</html>
