<?php
// Include database configuration
$db_config_path = __DIR__ . '/config/db.php';
if (!file_exists($db_config_path)) {
    error_log("[" . date('Y-m-d H:i:s') . "] Database configuration file missing: $db_config_path");
    die("<div class='alert alert-danger text-center'>Configuration file not found. Please contact support.</div>");
}
require_once $db_config_path;

// Start session if not already started
if (!session_id()) {
    session_start();
}

// Log page access
error_log("[" . date('Y-m-d H:i:s') . "] Using index.php version: b8c9d0e1-f2g3-4h5i-6j7k-8l9m0n1o2p3");

// Validate session
if (!isset($_SESSION['user_id'])) {
    header('Location: auth/login.php');
    exit;
}

// Database connection
$conn = new mysqli('localhost', 'root', '', 'SmartPay360');
if ($conn->connect_error) {
    error_log("Database connection failed: " . $conn->connect_error);
    die("<div class='alert alert-danger text-center'>Unable to connect to database. Please contact support.</div>");
}

$conn->set_charset('utf8mb4');

// Fetch dashboard metrics
$metrics = [
    'total_invoices' => 0,
    'pending_payments' => 0,
    'active_staff' => 0,
    'total_revenue' => 0.00
];

try {
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM invoices");
    $stmt->execute();
    $result = $stmt->get_result();
    $metrics['total_invoices'] = $result->fetch_assoc()['count'];
    $stmt->close();

    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM invoice_balances WHERE balance > 0");
    $stmt->execute();
    $result = $stmt->get_result();
    $metrics['pending_payments'] = $result->fetch_assoc()['count'];
    $stmt->close();

    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM staff WHERE status = 'active'");
    $stmt->execute();
    $result = $stmt->get_result();
    $metrics['active_staff'] = $result->fetch_assoc()['count'];
    $stmt->close();

    $stmt = $conn->prepare("SELECT SUM(amount) as total FROM payments WHERE payment_date >= ?");
    $start_date = date('Y-m-01');
    $stmt->bind_param('s', $start_date);
    $stmt->execute();
    $result = $stmt->get_result();
    $metrics['total_revenue'] = $result->fetch_assoc()['total'] ?? 0.00;
    $stmt->close();

    $recent_activities = [];
    $stmt = $conn->prepare("
        SELECT 'New invoice added' as activity, i.created_at, i.invoice_number as detail, b.branch_name
        FROM invoices i
        JOIN branches b ON i.branch_id = b.id
        WHERE i.created_at >= NOW() - INTERVAL 1 DAY
        UNION
        SELECT 'Payment received' as activity, p.created_at, p.reference_no as detail, NULL as branch_name
        FROM payments p
        WHERE p.created_at >= NOW() - INTERVAL 1 DAY
        UNION
        SELECT 'Staff updated' as activity, s.updated_at as created_at, s.first_name as detail, b.branch_name
        FROM staff s
        JOIN branches b ON s.branch_id = b.id
        WHERE s.updated_at >= NOW() - INTERVAL 1 DAY AND s.updated_at IS NOT NULL
        ORDER BY created_at DESC
        LIMIT 5
    ");
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $recent_activities[] = $row;
    }
    $stmt->close();
} catch (Exception $e) {
    error_log("Database error: " . $e->getMessage());
    echo "<div class='alert alert-danger text-center'>Error fetching dashboard data. Please try again later.</div>";
}

$conn->close();

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
if (!defined('APP_NAME')) define('APP_NAME', 'SmartPay360');
if (!defined('APP_VERSION')) define('APP_VERSION', '1.0.0');
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title><?php echo htmlspecialchars(APP_NAME); ?> - Dashboard</title>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" />
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet" />
  <link rel="icon" href="/smartpay360/assets/icons/favicon-32x32.png" />
  <link rel="manifest" href="/smartpay360/assets/icons/manifest.json" />
  <meta name="theme-color" content="#007bff" />

  <style>
    :root {
      --primary: #007bff;
      --dark: #1a1a1a;
      --light: #f4f7fa;
      --shadow: 0 8px 20px rgba(0, 0, 0, 0.15);
      --success: #28a745;
      --warning: #ffc107;
      --info: #17a2b8;
    }
    body {
      background: var(--light);
      font-family: 'Arial', sans-serif;
      min-height: 100vh;
      display: flex;
      flex-direction: column;
    }
    .hero-section {
      background: linear-gradient(135deg, var(--primary), var(--dark));
      border-radius: 15px;
      box-shadow: var(--shadow);
      padding: 60px 20px;
      text-align: center;
      color: white;
    }
    .hero-section h1 {
      font-size: 2.5rem;
    }
    .quick-card {
      border-radius: 10px;
      transition: transform 0.3s ease, box-shadow 0.3s ease;
    }
    .quick-card:hover {
      transform: translateY(-10px);
      box-shadow: var(--shadow);
    }
    footer {
      background-color: #0F3054;
      color: white;
      padding: 20px 0;
      margin-top: auto;
      text-align: center;
    }
    footer img {
      max-height: 40px;
      margin-bottom: 10px;
    }
    footer a {
      color: #FFDC00;
      text-decoration: underline;
    }

  </style>
</head>
<body>

  <!-- Hero Section -->
  <section class="hero-section mb-5">
    <div class="container text-center">
      <img src="/smartpay360/assets/icons/logo.png" alt="SmartPay360 Logo" class="img-fluid mb-4" style="max-width: 200px;">
      <h1 class="fw-bold">Welcome to <?php echo htmlspecialchars(APP_NAME); ?></h1>
      <p class="lead">Your ultimate solution for seamless payment and business management.</p>
    </div>
  </section>

  <!-- Quick Access -->
  <section class="quick-access mb-5 container">
    <h2 class="fw-bold mb-4 text-dark">Quick Access</h2>
    <div class="row g-4">
      <div class="col-md-3">
        <div class="card h-100 shadow-sm border-0 quick-card text-center">
          <div class="card-body">
            <i class="fas fa-file-invoice fa-3x text-primary mb-3"></i>
            <h5 class="card-title">Invoices</h5>
            <p class="text-muted">Manage all your invoices.</p>
            <a href="/smartpay360/modules/invoices/list.php" class="btn btn-outline-primary stretched-link">Go to Invoices</a>
          </div>
        </div>
      </div>
      <div class="col-md-3">
        <div class="card h-100 shadow-sm border-0 quick-card text-center">
          <div class="card-body">
            <i class="fas fa-money-check fa-3x text-success mb-3"></i>
            <h5 class="card-title">Payments</h5>
            <p class="text-muted">Track and process payments.</p>
            <a href="/smartpay360/modules/payments/list.php" class="btn btn-outline-success stretched-link">Go to Payments</a>
          </div>
        </div>
      </div>
      <div class="col-md-3">
        <div class="card h-100 shadow-sm border-0 quick-card text-center">
          <div class="card-body">
            <i class="fas fa-wallet fa-3x text-warning mb-3"></i>
            <h5 class="card-title">Petty Cash</h5>
            <p class="text-muted">Handle small transactions.</p>
            <a href="/smartpay360/modules/petty_cash/list.php" class="btn btn-outline-warning stretched-link">Go to Petty Cash</a>
          </div>
        </div>
      </div>
      <div class="col-md-3">
        <div class="card h-100 shadow-sm border-0 quick-card text-center">
          <div class="card-body">
            <i class="fas fa-bread-slice fa-3x text-info mb-3"></i>
            <h5 class="card-title">Bread Management</h5>
            <p class="text-muted">Manage bread loading & sales.</p>
            <a href="/smartpay360/modules/bread/loading.php" class="btn btn-outline-info stretched-link">Go to Bread</a>
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- Gustolix Footer -->
 <footer>
  <div class="container">
    <div class="logo-badge mx-auto mb-2">
      <img src="/smartpay360/assets/gustolix1.png" alt="Gustolix Logo">
    </div>
    <p class="mb-1">Designed & Powered by <strong>Gustolix Solutions</strong></p>
    <small>Innovative Business Automation • <a href="https://gustolix.co.ke" target="_blank">www.gustolix.co.ke</a></small>
  </div>
</footer>
